/*
 * Decompiled with CFR 0.152.
 */
package dan200.computercraft.core.filesystem;

import dan200.computercraft.api.filesystem.FileOperationException;
import dan200.computercraft.api.filesystem.WritableMount;
import dan200.computercraft.core.filesystem.FileMount;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.NonReadableChannelException;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Set;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WritableFileMount
extends FileMount
implements WritableMount {
    private static final Logger LOG = LoggerFactory.getLogger(WritableFileMount.class);
    private static final Set<OpenOption> WRITE_OPTIONS = Set.of(StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
    private static final Set<OpenOption> APPEND_OPTIONS = Set.of(StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.APPEND);
    protected final File rootFile;
    private final long capacity;
    private long usedSpace;

    public WritableFileMount(File rootFile, long capacity) {
        super(rootFile.toPath());
        this.rootFile = rootFile;
        this.capacity = capacity + 500L;
        this.usedSpace = this.created() ? WritableFileMount.measureUsedSpace(this.root) : 500L;
    }

    protected File resolveFile(String path) {
        return new File(this.rootFile, path);
    }

    private void create() throws FileOperationException {
        try {
            Files.createDirectories(this.root, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new FileOperationException("Access denied");
        }
    }

    @Override
    public long getRemainingSpace() {
        return Math.max(this.capacity - this.usedSpace, 0L);
    }

    @Override
    public long getCapacity() {
        return this.capacity - 500L;
    }

    @Override
    public boolean isReadOnly(String path) {
        File file = this.resolveFile(path);
        while (!file.exists()) {
            if (file.equals(this.rootFile)) {
                return false;
            }
            file = file.getParentFile();
        }
        return !file.canWrite();
    }

    @Override
    public void makeDirectory(String path) throws IOException {
        this.create();
        File file = this.resolveFile(path);
        if (file.exists()) {
            if (!file.isDirectory()) {
                throw new FileOperationException(path, "File exists");
            }
            return;
        }
        int dirsToCreate = 1;
        File parent = file.getParentFile();
        while (!parent.exists()) {
            ++dirsToCreate;
            parent = parent.getParentFile();
        }
        if (this.getRemainingSpace() < (long)dirsToCreate * 500L) {
            throw new FileOperationException(path, "Out of space");
        }
        if (file.mkdirs()) {
            this.usedSpace += (long)dirsToCreate * 500L;
        } else {
            throw new FileOperationException(path, "Access denied");
        }
    }

    @Override
    public void delete(String path) throws IOException {
        File file;
        if (path.isEmpty()) {
            throw new FileOperationException(path, "Access denied");
        }
        if (this.created() && (file = this.resolveFile(path)).exists()) {
            this.deleteRecursively(file);
        }
    }

    private void deleteRecursively(File file) throws IOException {
        if (file.isDirectory()) {
            String[] children;
            for (String aChildren : children = file.list()) {
                this.deleteRecursively(new File(file, aChildren));
            }
        }
        long fileSize = file.isDirectory() ? 0L : file.length();
        boolean success = file.delete();
        if (success) {
            this.usedSpace -= Math.max(500L, fileSize);
        } else {
            throw new IOException("Access denied");
        }
    }

    @Override
    public void rename(String source, String dest) throws FileOperationException {
        Path sourceFile = this.resolvePath(source);
        Path destFile = this.resolvePath(dest);
        if (!Files.exists(sourceFile, new LinkOption[0])) {
            throw new FileOperationException(source, "No such file");
        }
        if (Files.exists(destFile, new LinkOption[0])) {
            throw new FileOperationException(dest, "File exists");
        }
        if (destFile.startsWith(sourceFile)) {
            throw new FileOperationException(source, "Cannot move a directory inside itself");
        }
        try {
            Files.move(sourceFile, destFile, new CopyOption[0]);
        }
        catch (IOException e) {
            throw this.remapException(source, e);
        }
    }

    @Nullable
    private BasicFileAttributes tryGetAttributes(String path, Path resolved) throws FileOperationException {
        try {
            return Files.readAttributes(resolved, BasicFileAttributes.class, new LinkOption[0]);
        }
        catch (NoSuchFileException ignored) {
            return null;
        }
        catch (IOException e) {
            throw this.remapException(path, e);
        }
    }

    @Override
    public SeekableByteChannel openForWrite(String path) throws FileOperationException {
        this.create();
        Path file = this.resolvePath(path);
        BasicFileAttributes attributes = this.tryGetAttributes(path, file);
        if (attributes == null) {
            if (this.getRemainingSpace() < 500L) {
                throw new FileOperationException(path, "Out of space");
            }
        } else {
            if (attributes.isDirectory()) {
                throw new FileOperationException(path, "Cannot write to directory");
            }
            this.usedSpace -= Math.max(attributes.size(), 500L);
        }
        this.usedSpace += 500L;
        try {
            return new CountingChannel(Files.newByteChannel(file, WRITE_OPTIONS, new FileAttribute[0]), true);
        }
        catch (IOException e) {
            throw this.remapException(path, e);
        }
    }

    @Override
    public SeekableByteChannel openForAppend(String path) throws FileOperationException {
        this.create();
        Path file = this.resolvePath(path);
        BasicFileAttributes attributes = this.tryGetAttributes(path, file);
        if (attributes == null) {
            if (this.getRemainingSpace() < 500L) {
                throw new FileOperationException(path, "Out of space");
            }
        } else if (attributes.isDirectory()) {
            throw new FileOperationException(path, "Cannot write to directory");
        }
        try {
            return new CountingChannel(Files.newByteChannel(file, APPEND_OPTIONS, new FileAttribute[0]), false);
        }
        catch (IOException e) {
            throw this.remapException(path, e);
        }
    }

    private static long measureUsedSpace(Path path) {
        if (!Files.exists(path, new LinkOption[0])) {
            return 0L;
        }
        try {
            class CountingVisitor
            extends SimpleFileVisitor<Path> {
                long size;

                CountingVisitor() {
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                    this.size += 500L;
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    this.size += Math.max(attrs.size(), 500L);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) {
                    LOG.error("Error computing file size for {}", (Object)file, (Object)exc);
                    return FileVisitResult.CONTINUE;
                }
            }
            CountingVisitor visitor = new CountingVisitor();
            Files.walkFileTree(path, visitor);
            return visitor.size;
        }
        catch (IOException e) {
            LOG.error("Error computing file size for {}", (Object)path, (Object)e);
            return 0L;
        }
    }

    private class CountingChannel
    implements SeekableByteChannel {
        private final SeekableByteChannel channel;
        private final boolean canSeek;

        CountingChannel(SeekableByteChannel channel, boolean canSeek) {
            this.channel = channel;
            this.canSeek = canSeek;
        }

        @Override
        public int write(ByteBuffer b) throws IOException {
            int written;
            int toWrite = b.remaining();
            long newPosition = Math.addExact(this.channel.position(), (long)toWrite);
            long newBytes = newPosition - Math.max(500L, this.channel.size());
            if (newBytes > 0L) {
                long newUsedSpace = Math.addExact(WritableFileMount.this.usedSpace, newBytes);
                if (newUsedSpace > WritableFileMount.this.capacity) {
                    throw new IOException("Out of space");
                }
                WritableFileMount.this.usedSpace = newUsedSpace;
            }
            if ((written = this.channel.write(b)) != toWrite) {
                throw new IllegalStateException("Not all bytes were written");
            }
            assert (this.channel.position() == newPosition) : "Position is consistent";
            return written;
        }

        @Override
        public boolean isOpen() {
            return this.channel.isOpen();
        }

        @Override
        public void close() throws IOException {
            this.channel.close();
        }

        @Override
        public SeekableByteChannel position(long newPosition) throws IOException {
            if (!this.isOpen()) {
                throw new ClosedChannelException();
            }
            if (!this.canSeek) {
                throw new UnsupportedOperationException("File does not support seeking");
            }
            if (newPosition < 0L) {
                throw new IllegalArgumentException("Cannot seek before the beginning of the stream");
            }
            return this.channel.position(newPosition);
        }

        @Override
        public SeekableByteChannel truncate(long size) throws IOException {
            throw new UnsupportedOperationException("File cannot be truncated");
        }

        @Override
        public int read(ByteBuffer dst) throws ClosedChannelException {
            if (!this.channel.isOpen()) {
                throw new ClosedChannelException();
            }
            throw new NonReadableChannelException();
        }

        @Override
        public long position() throws IOException {
            return this.channel.position();
        }

        @Override
        public long size() throws IOException {
            return this.channel.size();
        }
    }
}

