-- SPDX-FileCopyrightText: 2023 The CC: Tweaked Developers
--
-- SPDX-License-Identifier: MPL-2.0

--[[- A parser for Lua programs and expressions.

> [!DANGER]
> This is an internal module and SHOULD NOT be used in your own code. It may
> be removed or changed at any time.

Most of the code in this module is automatically generated from the Lua grammar,
hence being mostly unreadable!

@local
]]

-- Lazily load our map of errors
local errors = setmetatable({}, {
    __index = function(self, key)
        setmetatable(self, nil)
        for k, v in pairs(require "cc.internal.syntax.errors") do self[k] = v end

        return self[key]
    end,
})

-- Everything below this line is auto-generated. DO NOT EDIT.

--- A lookup table of valid Lua tokens
local tokens = (function() return {} end)() -- Make tokens opaque to illuaminate. Nasty!
for i, token in ipairs({
    "WHILE", "UNTIL", "TRUE", "THEN", "SUB", "STRING", "SEMICOLON", "RETURN",
    "REPEAT", "POW", "OSQUARE", "OR", "OPAREN", "OBRACE", "NUMBER", "NOT",
    "NIL", "NE", "MUL", "MOD", "LT", "LOCAL", "LEN", "LE", "IN", "IF",
    "IDENT", "GT", "GE", "FUNCTION", "FOR", "FALSE", "EQUALS", "EQ", "EOF",
    "END", "ELSEIF", "ELSE", "DOTS", "DOT", "DO", "DIV", "CSQUARE", "CPAREN",
    "CONCAT", "COMMA", "COLON", "CBRACE", "BREAK", "AND", "ADD", "COMMENT",
    "ERROR",
}) do tokens[token] = i end
setmetatable(tokens, { __index = function(_, name) error("No such token " .. tostring(name), 2) end })

--- Read a integer with a given size from a string.
local function get_int(str, offset, size)
    if size == 1 then
        return str:byte(offset + 1)
    elseif size == 2 then
        local hi, lo = str:byte(offset + 1, offset + 2)
        return hi * 256 + lo
    elseif size == 3 then
        local b1, b2, b3 = str:byte(offset + 1, offset + 3)
        return b1 * 256 + b2 + b3 * 65536 -- Don't ask.
    else
        error("Unsupported size", 2)
    end
end

--[[ Error handling:

Errors are extracted from the current parse state in a two-stage process:
 - Run a DFA over the current state of the LR1 stack. For each accepting state,
   register a parse error.
 - Once all possible errors are found, pick the best of these and report it to
   the user.

This process is performed by a tiny register-based virtual machine. The bytecode
for this machine is stored in `error_program`, and the accompanying transition
table in `error_tbl`.

It would be more efficient to use tables here (`string.byte` is 2-3x slower than
a table lookup) or even define the DFA as a Lua program, however this approach
is much more space efficient - shaving off several kilobytes.

See https://github.com/let-def/lrgrep/ (namely ./support/lrgrep_runtime.ml) for
more information.
]]

local function is_same_line(context, previous, token)
    local prev_line = context.get_pos(previous)
    local tok_line = context.get_pos(token.s)
    return prev_line == tok_line and token.v ~= tokens.EOF
end

local function line_end_position(context, previous, token)
    if is_same_line(context, previous, token) then
        return token.s
    else
        return previous + 1
    end
end

local expr_tokens = {}
for _, v in pairs { tokens.STRING, tokens.NUMBER, tokens.TRUE, tokens.FALSE, tokens.NIL } do
  expr_tokens[v] = true
end

local error_messages = {
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 26
        if token.v == tokens.EQUALS then
        return errors.table_key_equals(token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 34
        if token.v == tokens.EQUALS then
        return errors.use_double_equals(token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 42
        if expr_tokens[token.v] then
      return errors.missing_table_comma(token.v, token.s, token.e, stack[stack_n + 2])
    end
    end,
    function(context, stack, stack_n, regs, token)
        local comma = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 52
        if token.v == tokens.CPAREN then
      return errors.trailing_call_comma(comma.s, comma.e, token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        local lp = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 60
        return errors.unclosed_brackets(lp.s, lp.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        local lp = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 62
        return errors.unclosed_brackets(lp.s, lp.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        local lp = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 64
        return errors.unclosed_brackets(lp.s, lp.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        local loc = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 69
        if token.v == tokens.DOT then
        return errors.local_function_dot(loc.s, loc.e, token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 77
        local end_pos = stack[stack_n + 2] -- Hack to get the last position
    if is_same_line(context, end_pos, token) then
        return errors.standalone_name(token.s)
    else
        return errors.standalone_name_call(end_pos)
    end
    end,
    function(context, stack, stack_n, regs, token)
        local start = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 88
        return errors.expected_then(start.s, start.e, line_end_position(context, stack[stack_n + 2], token))
    end,
    function(context, stack, stack_n, regs, token)
        local start = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        -- parse_errors.mlyl, line 116
        return errors.expected_end(start.s, start.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        local func = { s = stack[regs[2] + 1], e = stack[regs[2] + 2] }
        local loc = { s = stack[regs[3] + 1], e = stack[regs[3] + 2] }
        -- parse_errors.mlyl, line 120
        return errors.expected_end(loc.s, func.e, token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 124
        if token.v == tokens.END then
      return errors.unexpected_end(token.s, token.e)
    elseif token ~= tokens.EOF then
        return errors.expected_statement(token.v, token.s, token.e)
    end
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 134
        return errors.expected_function_args(token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 138
        return errors.expected_expression(token.v, token.s, token.e)
    end,
    function(context, stack, stack_n, regs, token)
        -- parse_errors.mlyl, line 142
        return errors.expected_var(token.v, token.s, token.e)
    end,
}
local error_program_start, error_program = 471, "\6\1\0\3\5\186\0\3\6B\0\3\0060\0\3\6(\0\3\6 \0\3\6\21\0\3\6\7\0\3\5\253\0\3\5\236\0\3\5\223\0\1\0\3\5\203\0\3\5\195\0\1\0\3\5\186\0\3\5\178\0\3\5\170\0\3\5\170\0\3\5\170\0\3\5\170\0\3\5\170\0\3\5\146\0\3\5\146\0\3\5\170\0\3\5\162\0\3\5\154\0\3\5\154\0\3\5\154\0\3\5\146\0\3\5\138\0\3\5p\0\3\5h\0\3\5`\0\3\5X\0\3\5P\0\3\4\158\0\3\5F\0\3\5B\0\3\0057\0\3\4\171\0\3\4\167\0\3\4\159\0\3\1\226\0\3\4\158\0\3\4\152\0\3\4\146\0\3\4\141\0\3\4\131\0\3\4{\0\3\4p\0\3\4O\0\3\4J\0\1\0\3\4E\0\1\0\3\4@\0\3\0043\0\3\4(\0\3\3\139\0\3\3\131\0\3\3\127\0\3\3w\0\3\3g\0\3\3{\0\3\3w\0\3\3w\0\3\3s\0\3\3o\0\3\2\233\0\3\3k\0\3\2\218\0\3\2\218\0\3\3g\0\3\3`\0\3\2\218\0\3\2\233\0\3\2\218\0\3\2\218\0\3\2\218\0\3\2\218\0\3\2\225\0\3\2\218\0\3\2\218\0\3\2\218\0\3\2\218\0\3\2Q\0\3\2I\0\3\2=\0\3\2E\0\3\2A\0\3\2\25\0\3\2\21\0\3\2=\0\3\0029\0\3\0025\0\3\0021\0\3\2\17\0\3\2-\0\3\2!\0\3\2)\0\3\2%\0\3\2!\0\3\2\29\0\3\2\25\0\3\2\21\0\3\2\17\0\3\2\r\0\3\2\t\0\3\2\5\0\3\2\1\0\3\2\1\0\3\1\253\0\3\1\249\0\3\1\245\0\3\1\245\0\3\1\235\0\3\1\241\0\3\1\235\0\3\1\226\0\5\0\0\3\4\136\0\3\6M\0\5\0\14\1\0\3\4@\0\1\0\3\4@\0\3\6M\0\3\6U\0\3\6U\0\3\1\249\0\3\6^\0\3\6b\0\3\4J\0\3\6f\0\3\2)\0\3\4p\0\3\2\21\0\3\2\25\0\3\2\17\0\3\2I\0\3\2=\0\3\0029\0\3\2-\0\3\6j\0\3\2!\0\3\2%\0\3\6\142\0\3\2A\0\3\6\142\0\3\2\21\0\5\0\5\1\0\3\4@\0\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\6\234\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\202\0\3\6\202\0\3\6\226\0\3\6\218\0\3\6\210\0\3\6\210\0\3\6\210\0\3\6\202\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\3\2\218\0\3\2\218\0\3\2\218\0\3\3w\0\5\0\190\3\6\149\0\3\3g\0\3\7.\0\5\0\8\3\7\182\0\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\6\234\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\202\0\3\6\202\0\3\6\226\0\3\6\218\0\3\6\210\0\3\6\210\0\3\6\210\0\3\6\202\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\5\1\3\3\6\149\0\3\3w\0\3\7\189\0\3\7.\0\3\7\182\0\3\2\218\0\3\7\193\0\3\3\131\0\3\7\197\0\3\7\r\0\5\0\27\1\0\3\7\29\0\3\7\211\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\4J\0\1\0\3\4@\0\3\6\234\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\202\0\3\6\202\0\3\6\226\0\3\6\218\0\3\6\210\0\3\6\210\0\3\6\210\0\3\6\202\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\3\2\218\0\3\2\218\0\3\2\218\0\3\3w\0\3\7\201\0\3\2I\0\3\6b\0\3\6M\0\5\1f\3\6\149\0\3\1\249\0\4\2\0\0\5\0\31\1\0\3\4E\0\4\4\0\1\6\4\5\0\1\6\4\8\0\0\6\4\12\0\0\6\1\0\3\7\219\0\3\7\243\0\3\7\239\0\3\7\235\0\3\7\228\0\1\0\3\7\219\0\4\12\0\0\5\0\214\3\8\1\0\4\12\0\0\3\8$\0\4\r\0\0\6\4\14\0\0\6\4\15\0\0\6\1\0\3\4@\0\1\0\3\4E\0\6\3\7\211\0\3\2I\0\5\0\18\6\3\7\201\0\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\4J\0\3\6\234\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\202\0\3\6\202\0\3\6\226\0\3\6\218\0\3\6\210\0\3\6\210\0\3\6\210\0\3\6\202\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\3\2\218\0\3\2\218\0\3\2\218\0\3\3w\0\3\6b\0\5\1\233\3\6\149\0\3\7\235\0\5\0\30\6\3\7\239\0\1\0\3\8+\0\4\1\0\0\5\0B\6\4\1\0\0\3\0080\0\4\1\0\0\3\0084\0\4\1\0\0\3\2=\0\4\1\0\0\3\0088\0\1\0\3\4@\0\3\2%\0\3\2=\0\3\2\21\0\4\1\0\0\5\0T\6\4\1\0\0\3\8<\0\4\1\0\0\3\8@\0\4\1\0\0\3\8D\0\4\1\0\0\3\8H\0\4\1\0\0\3\8L\0\4\12\0\0\4\n\0\1\6\4\r\0\0\3\8P\0\4\14\0\0\3\8Z\0\3\8\1\0\3\7\243\0\3\7\228\0\5\1\19\1\0\3\7\219\0\3\8^\0\4\1\0\0\4\0\0\0\5\0008\1\0\3\8f\0\4\1\0\0\1\0\3\4@\0\4\1\0\0\1\0\3\8o\0\3\7\243\0\4\12\0\0\5\0%\3\8t\0\4\12\0\0\3\7\243\0\4\12\0\0\3\8z\0\4\12\0\0\1\0\3\7\219\0\3\7\243\0\3\8\128\0\4\12\0\0\5\0\224\3\8\132\0\3\1\235\0\3\1\249\0\5\0c\1\0\3\4E\0\3\8\136\0\3\2\t\0\3\8\150\0\3\0021\0\1\0\3\5\186\0\3\0060\0\3\6(\0\3\6 \0\3\6\21\0\1\0\3\5\186\0\3\4O\0\5\1L\3\6B\0\4\1\0\0\6\4\1\0\0\3\8\157\0\4\1\0\0\3\8\165\0\4\1\0\0\3\4p\0\4\1\0\0\3\2%\0\4\1\0\0\3\2I\0\4\1\0\0\3\8\192\0\4\1\0\0\3\t \0\4\1\0\0\3\t\139\0\4\1\0\0\3\t\202\0\4\1\0\0\3\n\17\0\4\1\0\0\3\nT\0\4\4\0\1\4\1\0\0\6\4\6\0\1\4\1\0\0\6\4\t\0\1\4\1\0\0\6\4\2\0\0\4\1\0\0\4\0\0\0\3\6U\0\4\5\0\1\4\2\0\0\4\1\0\0\4\0\0\0\6\3\3s\0\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\6\234\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\202\0\3\6\202\0\3\6\226\0\3\6\218\0\3\6\210\0\3\6\210\0\3\6\210\0\3\6\202\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\3\2\218\0\3\2\218\0\3\2\218\0\3\3w\0\3\6\142\0\5\2L\3\6\149\0\3\2\233\0\3\n\227\0\3\0057\0\3\4\167\0\1\0\3\n\234\0\4\r\0\0\5\0V\6\4\n\0\1\6\3\7\243\0\5\0=\3\8t\0\3\7\243\0\3\8z\0\1\0\3\7\219\0\3\7\243\0\3\8\128\0\5\0\230\3\8\132\0\1\0\3\7\219\0\3\7\243\0\3\7\239\0\3\7\235\0\3\7\228\0\1\0\3\7\219\0\5\1U\3\8\1\0\4\t\0\1\6\3\8\157\0\3\8\165\0\3\8\192\0\3\t \0\3\t\139\0\3\t\202\0\3\n\17\0\3\nT\0\1\0\3\n\234\0\3\n\239\0\5\0\198\6\4\2\0\0\3\6U\0\4\5\0\1\4\2\0\0\6\4\6\0\1\6\1\0\3\n\246\0\1\0\3\n\255\0\3\11\7\0\3\11\11\0\3\4\131\0\1\0\3\5\186\0\3\11\15\0\5\0\192\3\0060\0\3\0084\0\3\8^\0\5\0m\1\0\3\8f\0\1\0\3\4@\0\3\2%\0\3\2=\0\3\2\21\0\5\0\148\6\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\6\234\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\202\0\3\6\226\0\3\6\218\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\5\2\143\3\6\149\0\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\6\234\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\202\0\3\6\202\0\3\6\226\0\3\6\218\0\3\6\202\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\5\2\212\3\6\149\0\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\5\2c\3\6\149\0\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\6\234\0\3\6\218\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\5\3\23\3\6\149\0\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\6\218\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\5\3 \3\6\149\0\1\0\3\7\29\0\3\7\r\0\1\0\3\7\4\0\1\0\3\6\251\0\1\0\3\6\242\0\3\6\234\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\226\0\3\6\202\0\3\6\202\0\3\6\226\0\3\6\218\0\3\6\210\0\3\6\210\0\3\6\210\0\3\6\202\0\3\6\194\0\3\6\186\0\3\6\178\0\3\6\178\0\3\6\170\0\3\6\162\0\3\6\154\0\3\2\218\0\3\2\218\0\3\2\218\0\3\7\182\0\3\7.\0\3\3w\0\5\3\163\3\6\149\0\4\7\0\1\6\5\0\11\3\11\25\0\2\1\0\1\0\3\11\30\0\4\n\0\1\3\4\158\0\3\7\243\0\3\11'\0\4\12\0\0\1\0\3\n\246\0\4\3\0\1\6\4\11\1\2\6\3\11.\0\5\0\136\3\0112\0\3\7\243\0\3\0116\0\3\11.\0"
local error_tbl_ks, error_tbl_vs, error_tbl = 1, 2, "\0\0\199\1\0\147\2\0\0\3\0\195\0\0\0\5\1K\0\0\0\7\1G\0\0\0\t\0\207\0\0\0\11\1C\0\0\0\r\1?\0\0\0\15\0\223\16\0\187\17\0\213\18\1\211\19\0\167\20\1\207\21\0\0\22\1\203\23\0\179\24\1\163\25\1\199\26\0\11\27\0'\28\1;\29\0017\30\0013\31\0\151 \1/!\1+\"\1'\21\1\222$\1#%\1\31\0\0\0'\1\27(\1\23)\1\19*\0\27\0\0\0,\0o\0\0\0.\0k\0\0\0000\0g\0\0\0002\0c\0\0\0004\0_\0\0\0006\0[\0\0\0008\0W\0\0\0:\0S\0\0\0<\0O\0\0\0>\0K\0\0\0@\0G\0\0\0B\0C\0\0\0D\0?\0\0\0F\0;G\0\235H\0\213I\1OJ\0+K\0wL\0\179M\1\15N\0sO\0\0P\0\231Q\0\0R\0\0S\1\11T\1\7U\1\3V\0\255W\0\251X\0\27Y\0\0R\2\229[\0\131\\\0\227R\0\0^\0\127_\0\219`\1\195a\1\191b\1\187c\1\183d\0#e\0\175f\0\247g\0\31h\0\243i\0\239j\0\135k\0\7l\0'm\1\159n\1\155o\1\151p\1\147q\0\199r\0\147n\2Mt\0{u\0\0v\0\183w\0001x\0\23y\0'z\1\143{\0\159|\1\139b\3\135~\1\135\127\0\183\128\0\155b\4/\130\0\135\131\0\19\132\0\147\133\0\139\134\1\131u\4\163\136\0\135\137\0\15\138\0\143\139\0005\140\0'\141\1\127\142\0\183\143\0\163\144\0\187\145\0\0\146\1\179\147\0\0\148\1\175\149\0\23\150\0'\151\1{\152\0\183\153\0\171b\5\232\155\0\135H\5x\157\0\135\158\0\7\159\0'\160\1w\131\5>\162\0\135\163\0\7\164\0'\165\1s\166\0\0\137\5>\168\0\0\169\0\7\170\0'\171\1o\172\0\1\173\0'\174\1k\175\1g\176\1c\177\1_\178\1[\179\0\0\180\0\203\144\6\17\182\1W\183\0\183\184\1\171\185\1\167\186\0\191\187\1S\188\0\0\189\0\0\190\0\0\191\0\0\192\0\0\3\0\0n\5\134\129\5J\6\2\202b\6Q\8\2p\t\2Z\n\2j\135\5J\12\2\206u\7\205\14\2\210\144\7\224\8\8Vb\8\161\16\8\140}\5\130" .. ("\0"):rep(15) .. "\17\8V\0\0\0\0\0\0\0\0\0\0\0\0H\8\174\0\0\0\0\0\0\0\0\0\0\0\0#\2jZ\11#\0\0\0&\2\214]\11#\0\0\0\0\0\0\0\0\0+\2\170\0\0\0-\2\166\0\0\0/\2\162\0\0\0001\2\158\26\4Z3\2\154\0\0\0005\2\150\0\0\0007\2\146\0\0\0009\2\142\0\0\0;\2\138\0\0\0=\2\134\0\0\0?\2\130\0\0\0A\2~\0\0\0C\2zn\8\188E\2v\0\0\0}\11#H\2p\0\0\0J\2\174\181\5~\0\0\0\8\3\6\t\2\240\n\3\0H\8V\0\0\0\0\0\0}\8\184" .. ("\0"):rep(18) .. "Z\2\198\0\0\0\0\0\0]\2\194\0\0\0\0\0\0\0\0\0\0\0\0b\2`\0\0\0\154\11#\0\0\0\0\0\0\0\0\0#\3\0\0\0\0\0\0\0I\6>\0\0\0\0\0\0n\2\190\26\5\215+\3@I\7\253-\3<s\2\186/\0038\0\0\0001\0034\0\0\0003\0030w\8\1465\3,\0\0\0007\3(}\2\1829\3$\181\11#;\3 \129\2d=\3\28\0\0\0?\3\24\0\0\0A\3\20\135\2dC\3\16q\0\0E\3\12\181\8\180j\6:H\3\6\0\6\138J\3Dx\4f\0\0\0j\7\249" .. ("\0"):rep(24) .. "\131\4b\0\0\0\0\0\0\0\0\0Z\3\\~\6>\137\4^]\3X\139\4j\0\0\0\0\0\0~\7\253b\2\246\26\6t\0\0\0\0\0\0\3\3\170\0\0\0\149\4f\6\4\8\0\0\0\8\3\164\26\8\14\n\3\158n\3T\12\4\12\181\2\178\14\4\16\183\0\0s\3P\17\3\174\0\0\0\0\0\0\0\0\0\21\4$\0\0\0\0\0\0k\5\211\0\0\0}\3L\0\0\0\172\4T\0\0\0\129\2\250\0\0\0\0\0\0\0\0\0\168\6>#\3\158\135\2\250x\5\219&\4\20\0\0\0\168\7\253\0\0\0\0\0\0+\3\232\0\0\0-\3\228\0\0\0/\3\224\182\6>1\3\220\0\0\0003\3\216\0\0\0005\3\212\182\7\2537\3\208\190\6>9\3\204\0\0\0;\3\200\0\0\0=\3\196\190\7\253?\3\192\0\0\0A\3\188\149\5\219C\3\184\0\0\0E\3\180\0\0\0\0\0\0H\3\164\0\0\0J\3\236\158\5\211\0\0\0\0\0\0\0\0\0\0\0\0\163\5\211\0\0\0\181\3H\0\0\0\0\0\0\0\0\0\169\5\211q\6\138\0\0\0\0\0\0Z\4\4\0\0\0\0\0\0]\4\0x\6\128\0\0\0q" .. ("\0"):rep(20) .. "x\8\26\0\0\0\131\6|\0\0\0\0\0\0\0\0\0\0\0\0n\3\252\137\6x\0\0\0\139\6\132\131\8\22s\3\248\0\0\0u\4\28v\3\148\0\0\0\137\8\18\0\0\0\139\8\30\149\6\128\0\0\0}\3\244\0\0\0\127\4\24\0\0\0\129\3\152\0\0\0\0\0\0\149\8\26\0\0\0\3\4\203\135\3\152\0\0\0\6\5#\0\0\0\8\4\197\t\4\175\n\4\191\142\3\148\12\5'\0\0\0\14\5+\172\6n" .. ("\0"):rep(15) .. "\152\0\0\0\0\0\0\0\0\172\8\8" .. ("\0"):rep(30) .. "#\4\191\0\0\0\0\0\0&\5/\0\0\0\0\0\0\0\0\0\0\0\0+\5\3\0\0\0-\4\255\0\0\0/\4\251\0\0\0001\4\247\181\3\2403\4\243\183\4 5\4\239\0\0\0007\4\235\0\0\0009\4\231\0\0\0;\4\227\0\0\0=\4\223\0\0\0?\4\219\0\0\0A\4\215\0\0\0C\4\211\0\0\0E\4\207\0\0\0\0\0\0H\4\197\0\0\0J\5\7" .. ("\0"):rep(45) .. "Z\5\31\0\0\0\0\0\0]\5\27\0\0\0\0\0\0\0\0\0\0\0\0b\4\181\0\0\0\0\0\0\0\0\0\3\7\178\0\0\0\0\0\0\6\7\162\0\0\0\8\7H\t\0072\n\7Bn\5\23\12\7\166\0\0\0\14\7\170\0\0\0s\5\19" .. ("\0"):rep(27) .. "}\5\15\0\0\0\0\0\0\0\0\0\129\4\185\8\t\168\t\t\146\n\t\162\0\0\0#\7B\135\4\185\0\0\0&\7\174\0\0\0\0\0\0\0\0\0\0\0\0+\7\130\0\0\0-\7~\0\0\0/\7z\0\0\0001\7v\0\0\0003\7r\0\0\0005\7n\0\0\0007\7j\0\0\0009\7f#\t\162;\7b\0\0\0=\7^\0\0\0?\7Z\0\0\0A\7V\0\0\0C\7R\0\0\0E\7N\0\0\0\0\0\0H\7H\0\0\0J\7\134\8\8\218\t\8\196\n\8\212\0\0\0\0\0\0\0\0\0\0\0\0\181\5\11\0\0\0\183\0053" .. ("\0"):rep(15) .. "Z\7\158\0\0\0\0\0\0]\7\154\0\0\0H\t\168\0\0\0J\t\174b\0078\0\0\0\0\0\0\0\0\0#\8\212" .. ("\0"):rep(21) .. "n\7\150\0\0\0\0\0\0Z\t\198\0\0\0s\7\146]\t\194\0\0\0003\t\0\0\0\0005\8\252b\t\1527\8\248\0\0\0\0\0\0}\7\142;\8\244\0\0\0=\8\240\129\7<?\8\236\0\0\0A\8\232n\t\190C\8\228\135\7<E\8\224\0\0\0s\t\186H\8\218\0\0\0J\t\4\0\0\0\0\0\0\8\t=\t\t'\n\t7\0\0\0}\t\182\0\0\0\0\0\0\0\0\0\129\t\156\0\0\0\0\0\0\0\0\0\0\0\0Z\t\28\135\t\156\0\0\0]\t\24\0\0\0\0\0\0\0\0\0\0\0\0b\8\202" .. ("\0"):rep(15) .. "#\t7" .. ("\0"):rep(15) .. "n\t\20\0\0\0+\tk\0\0\0\181\7\138s\t\16\183\0\0\0\0\0\0\0\0\0\0\0003\tg\0\0\0005\tc\0\0\0007\t_}\t\0129\t[\0\0\0;\tW\129\8\206=\tS\0\0\0?\tO\0\0\0A\tK\135\8\206C\tG\181\t\178E\tC\0\0\0\0\0\0H\t=\0\0\0J\to\8\t\231\t\t\209\n\t\225" .. ("\0"):rep(18) .. "\8\n.\t\n\24\n\n(\0\0\0\0\0\0\0\0\0Z\t\135\0\0\0\0\0\0]\t\131\0\0\0\0\0\0\0\0\0\0\0\0b\t-\0\0\0\0\0\0\0\0\0#\t\225" .. ("\0"):rep(21) .. "n\t\127#\n(\181\t\8\0\0\0\0\0\0s\t{\0\0\0\0\0\0003\t\241" .. ("\0"):rep(18) .. "}\tw\0\0\0003\n4\0\0\0\129\t1" .. ("\0"):rep(15) .. "\135\t1E\t\237\0\0\0\0\0\0H\t\231\0\0\0J\t\245" .. ("\0"):rep(18) .. "H\n.\0\0\0J\n8" .. ("\0"):rep(18) .. "Z\n\r\0\0\0\0\0\0]\n\t\0\0\0\0\0\0\0\0\0\0\0\0b\t\215Z\nP\0\0\0\0\0\0]\nL\0\0\0\0\0\0\0\0\0\0\0\0b\n\30\0\0\0\0\0\0n\n\5\0\0\0\0\0\0\0\0\0\181\tss\n\1\0\0\0\0\0\0\0\0\0n\nH\0\0\0\0\0\0\0\0\0\0\0\0s\nD}\t\253\0\0\0\0\0\0\0\0\0\129\t\219\0\0\0\0\0\0\0\0\0\0\0\0}\n@\135\t\219\0\0\0\0\0\0\129\n\"" .. ("\0"):rep(15) .. "\135\n\"\0\0\0\6\n\203\0\0\0\8\nq\t\n[\n\nk\0\0\0\12\n\207\0\0\0\14\n\211" .. ("\0"):rep(54) .. "!\n\215\0\0\0#\nk\0\0\0\0\0\0&\n\223\0\0\0\0\0\0\181\t\249\0\0\0+\n\171\0\0\0-\n\167\0\0\0/\n\163\0\0\0001\n\159\181\n<3\n\155\0\0\0005\n\151\0\0\0007\n\147\0\0\0009\n\143\0\0\0;\n\139\0\0\0=\n\135\0\0\0?\n\131\0\0\0A\n\127\0\0\0C\n{\0\0\0E\nw\0\0\0\0\0\0H\nq\0\0\0J\n\175" .. ("\0"):rep(21) .. "R\n\219" .. ("\0"):rep(21) .. "Z\n\199\0\0\0\0\0\0]\n\195\0\0\0\0\0\0\0\0\0\0\0\0b\na" .. ("\0"):rep(33) .. "n\n\191\0\0\0\0\0\0\0\0\0\0\0\0s\n\187" .. ("\0"):rep(27) .. "}\n\183\0\0\0\0\0\0\0\0\0\129\ne" .. ("\0"):rep(15) .. "\135\ne" .. ("\0"):rep(129) .. "\179\n\215\0\0\0\181\n\179"

local function handle_error(context, stack, stack_n, token, token_start, token_end)
    -- Run our error handling virtual machine.
    local pc, top, registers, messages = error_program_start, stack_n, {}, {}
    while true do
        local instruction = error_program:byte(pc + 1)
        if instruction == 1 then -- Store
            registers[error_program:byte(pc + 2) + 1] = top
            pc = pc + 2
        elseif instruction == 2 then -- Move
            registers[error_program:byte(pc + 2) + 1] = registers[error_program:byte(pc + 3) + 1]
            pc = pc + 3
        elseif instruction == 3 then -- Pop one item from the stack and jump
            if top > 1 then top = top - 3 end
            pc = get_int(error_program, pc + 1, 3)
        elseif instruction == 4 then -- Accept
            local clause, _, count = error_program:byte(pc + 2, pc + 4)
            local accept = { clause + 1 }
            for i = 1, count do accept[i + 1] = registers[count - i + 1] end
            messages[#messages + 1] = accept

            pc = pc + 4
        elseif instruction == 5 then -- Match
            local hi, lo = error_program:byte(pc + 2, pc + 3)
            local lr1 = stack[top] - 1

            local offset = (hi * 256 + lo + lr1) * (error_tbl_ks + error_tbl_vs)
            if offset + error_tbl_ks + error_tbl_vs <= #error_tbl and
                get_int(error_tbl, offset, error_tbl_ks) == lr1 then
                pc = get_int(error_tbl, offset + error_tbl_ks, error_tbl_vs)
            else
                pc = pc + 3
            end
        elseif instruction == 6 then -- Halt
            break
        else
            error("Illegal instruction while handling errors " .. tostring(instruction))
        end
    end

    -- Sort the list to ensure earlier patterns are used first.
    table.sort(messages, function(a, b) return a[1] < b[1] end)

    -- Then loop until we find an error message which actually works!
    local t = { v = token, s = token_start, e = token_end }
    for i = 1, #messages do
        local action = messages[i]
        local message = error_messages[action[1]](context, stack, stack_n, action, t)
        if message then
            context.report(message)
            return false
        end
    end

    context.report(errors.unexpected_token, token, token_start, token_end)
    return false
end

--- The list of productions in our grammar. Each is a tuple of `terminal * production size`.
local productions = {
    { 53, 1 }, { 52, 1 }, { 81, 1 }, { 81, 1 }, { 80, 3 }, { 79, 1 },
    { 79, 1 }, { 79, 1 }, { 79, 1 }, { 79, 1 }, { 79, 1 }, { 79, 1 },
    { 79, 1 }, { 79, 4 }, { 78, 2 }, { 78, 4 }, { 77, 3 }, { 77, 1 },
    { 77, 1 }, { 76, 1 }, { 76, 3 }, { 76, 3 }, { 76, 3 }, { 76, 3 },
    { 76, 3 }, { 76, 3 }, { 76, 3 }, { 76, 3 }, { 76, 3 }, { 76, 3 },
    { 76, 3 }, { 76, 3 }, { 76, 3 }, { 76, 3 }, { 75, 1 }, { 75, 3 },
    { 75, 2 }, { 75, 2 }, { 75, 2 }, { 74, 1 }, { 74, 3 }, { 74, 3 },
    { 73, 0 }, { 73, 2 }, { 73, 3 }, { 73, 1 }, { 73, 2 }, { 72, 1 },
    { 72, 3 }, { 72, 4 }, { 71, 2 }, { 70, 2 }, { 69, 0 }, { 69, 2 },
    { 69, 3 }, { 68, 0 }, { 68, 5 }, { 67, 0 }, { 67, 1 }, { 66, 0 },
    { 66, 1 }, { 65, 1 }, { 65, 3 }, { 64, 1 }, { 64, 3 }, { 63, 1 },
    { 63, 3 }, { 62, 1 }, { 62, 3 }, { 61, 1 }, { 61, 3 }, { 61, 1 },
    { 60, 3 }, { 60, 3 }, { 60, 5 }, { 60, 4 }, { 60, 6 }, { 60, 8 },
    { 60, 9 }, { 60, 11 }, { 60, 7 }, { 60, 2 }, { 60, 4 }, { 60, 6 },
    { 60, 5 }, { 60, 1 }, { 59, 2 }, { 58, 3 }, { 57, 0 }, { 57, 1 },
    { 57, 3 }, { 56, 1 }, { 56, 3 }, { 56, 5 }, { 55, 1 }, { 55, 1 },
    { 54, 1 },
}

local f = false

--[[- The state machine used for our grammar.

Most LR(1) parsers will encode the transition table in a compact binary format,
optimised for space and fast lookups. However, without access to built-in
bitwise operations, this is harder to justify in Lua. Instead, the transition
table is a 2D lookup table of `action = transitions[state][value]`, where
`action` can be one of the following:

 - `action = false`: This transition is undefined, and thus a parse error. We
   use this (rather than nil) to ensure our tables are dense, and thus stored as
   arrays rather than maps.

 - `action > 0`: Shift this terminal or non-terminal onto the stack, then
   transition to `state = action`.

 - `action < 0`: Apply production `productions[-action]`. This production is a
   tuple composed of the next state and the number of values to pop from the
   stack.
]]
local transitions = {
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 2, f, f, f, f, f, f, f, f, f, 4, f, 189 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 3 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -51 },
    { 5, -43, f, f, f, f, f, 111, 114, f, f, f, 9, f, f, f, f, f, f, f, f, 118, f, f, f, 130, 16, f, f, 143, 153, f, f, f, -43, -43, -43, -43, f, f, 173, f, f, f, f, f, f, f, 176, f, f, f, f, 32, f, f, f, f, f, 178, 180, f, 181, f, f, f, f, f, f, f, f, 186, 187, f, f, f, f, 188 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 107, f, 41, 42 },
    { -9, -9, f, -9, -9, f, -9, -9, -9, -9, f, -9, -9, f, f, f, f, -9, -9, -9, -9, -9, f, -9, f, -9, -9, -9, -9, -9, -9, f, f, -9, -9, -9, -9, -9, f, f, -9, -9, -9, -9, -9, -9, f, -9, -9, -9, -9 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 106, f, f, 41, 42 },
    { -13, -13, f, -13, -13, f, -13, -13, -13, -13, f, -13, -13, f, f, f, f, -13, -13, -13, -13, -13, f, -13, f, -13, -13, -13, -13, -13, -13, f, f, -13, -13, -13, -13, -13, f, f, -13, -13, -13, -13, -13, -13, f, -13, -13, -13, -13 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 104, f, 41, 42 },
    { f, f, 6, f, 7, 8, f, f, f, f, 11, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 93, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, -89, f, f, f, f, f, 32, f, 96, 102, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 101, f, 41, 42 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 89, f, 41, 42 },
    { -12, -12, f, -12, -12, f, -12, -12, -12, -12, f, -12, -12, f, f, f, f, -12, -12, -12, -12, -12, f, -12, f, -12, -12, -12, -12, -12, -12, f, f, -12, -12, -12, -12, -12, f, f, -12, -12, -12, -12, -12, -12, f, -12, -12, -12, -12 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 88, f, f, 41, 42 },
    { -8, -8, f, -8, -8, f, -8, -8, -8, -8, f, -8, -8, f, f, f, f, -8, -8, -8, -8, -8, f, -8, f, -8, -8, -8, -8, -8, -8, f, f, -8, -8, -8, -8, -8, f, f, -8, -8, -8, -8, -8, -8, f, -8, -8, -8, -8 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 87, f, f, 41, 42 },
    { -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97 },
    { f, f, f, f, f, f, f, f, f, f, f, f, 18, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 27 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, f, f, f, f, f, f, f, f, f, f, 19, f, f, f, f, -58, f, f, f, f, f, f, f, f, f, 20, f, f, f, f, f, f, f, f, f, f, 21, f, 24, f, f, f, f, f, f, f, f, f, f, f, f, f, 26 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -4, f, -4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -3, f, -3 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -59, f, 22 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, f, f, f, f, f, f, f, f, f, f, 19, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 20, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 23 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -63, f, -63 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 25 },
    { -5, f, f, f, f, f, f, -5, -5, f, f, f, -5, f, f, f, f, f, f, f, f, -5, f, f, f, -5, -5, f, f, -5, -5, f, f, f, f, -5, f, f, f, f, -5, f, f, f, f, f, f, f, -5 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -62, f, -62 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 28, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 29 },
    { -14, -14, f, -14, -14, f, -14, -14, -14, -14, f, -14, -14, f, f, f, f, -14, -14, -14, -14, -14, f, -14, f, -14, -14, -14, -14, -14, -14, f, f, -14, -14, -14, -14, -14, f, f, -14, -14, -14, -14, -14, -14, f, -14, -14, -14, -14 },
    { -10, -10, f, -10, -10, f, -10, -10, -10, -10, f, -10, -10, f, f, f, f, -10, -10, -10, -10, -10, f, -10, f, -10, -10, -10, -10, -10, -10, f, f, -10, -10, -10, -10, -10, f, f, -10, -10, -10, -10, -10, -10, f, -10, -10, -10, -10 },
    { -11, -11, f, -11, -11, f, -11, -11, -11, -11, f, -11, -11, f, f, f, f, -11, -11, -11, -11, -11, f, -11, f, -11, -11, -11, -11, -11, -11, f, f, -11, -11, -11, -11, -11, f, f, -11, -11, -11, -11, -11, -11, f, -11, -11, -11, -11 },
    { -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48, -48 },
    { -7, -7, f, -7, -7, f, -7, -7, -7, -7, f, -7, -7, f, f, f, f, -7, -7, -7, -7, -7, f, -7, f, -7, -7, -7, -7, -7, -7, f, f, -7, -7, -7, -7, -7, f, f, -7, -7, -7, -7, -7, -7, f, -7, -7, -7, -7 },
    { -6, -6, f, -6, -6, 35, -6, -6, -6, -6, 36, -6, 73, 10, f, f, f, -6, -6, -6, -6, -6, f, -6, f, -6, -6, -6, -6, -6, -6, f, f, -6, -6, -6, -6, -6, f, 80, -6, -6, -6, -6, -6, -6, 82, -6, -6, -6, -6, f, f, f, f, f, f, 84, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 86 },
    { -18, -18, f, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18, f, f, f, -18, -18, -18, -18, -18, f, -18, f, -18, -18, -18, -18, -18, -18, f, f, -18, -18, -18, -18, -18, f, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18, -18 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 43, f, 41, 42 },
    { -70, -70, f, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, f, f, f, -70, -70, -70, -70, -70, f, -70, f, -70, -70, -70, -70, -70, -70, f, f, -70, -70, -70, -70, -70, f, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70, -70 },
    { -20, -20, f, -20, -20, f, -20, -20, -20, 39, f, -20, -20, f, f, f, f, -20, -20, -20, -20, -20, f, -20, f, -20, -20, -20, -20, -20, -20, f, f, -20, -20, -20, -20, -20, f, f, -20, -20, -20, -20, -20, -20, f, -20, -20, -20, -20 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 40, f, f, 41, 42 },
    { -36, -36, -36, -36, -36, -36, -36, -36, -36, 39, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36, -36 },
    { -72, -72, f, -72, -72, -72, -72, -72, -72, -72, -72, -72, -72, -72, f, f, f, -72, -72, -72, -72, -72, f, -72, f, -72, -72, -72, -72, -72, -72, f, f, -72, -72, -72, -72, -72, f, -72, -72, -72, -72, -72, -72, -72, -72, -72, -72, -72, -72 },
    { -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35, -35 },
    { f, f, f, f, 44, f, f, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, f, 50, 72, f, 56, f, f, f, f, 70, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 45, f, 41, 42 },
    { -24, -24, f, -24, -24, f, -24, -24, -24, f, f, -24, -24, f, f, f, f, -24, 46, 48, -24, -24, f, -24, f, -24, -24, -24, -24, -24, -24, f, f, -24, -24, -24, -24, -24, f, f, -24, 50, -24, -24, -24, -24, f, -24, -24, -24, -24 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 47, f, 41, 42 },
    { -25, -25, f, -25, -25, f, -25, -25, -25, f, f, -25, -25, f, f, f, f, -25, -25, -25, -25, -25, f, -25, f, -25, -25, -25, -25, -25, -25, f, f, -25, -25, -25, -25, -25, f, f, -25, -25, -25, -25, -25, -25, f, -25, -25, -25, -25 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 49, f, 41, 42 },
    { -27, -27, f, -27, -27, f, -27, -27, -27, f, f, -27, -27, f, f, f, f, -27, -27, -27, -27, -27, f, -27, f, -27, -27, -27, -27, -27, -27, f, f, -27, -27, -27, -27, -27, f, f, -27, -27, -27, -27, -27, -27, f, -27, -27, -27, -27 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 51, f, 41, 42 },
    { -26, -26, f, -26, -26, f, -26, -26, -26, f, f, -26, -26, f, f, f, f, -26, -26, -26, -26, -26, f, -26, f, -26, -26, -26, -26, -26, -26, f, f, -26, -26, -26, -26, -26, f, f, -26, -26, -26, -26, -26, -26, f, -26, -26, -26, -26 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 53, f, 41, 42 },
    { -22, -22, f, -22, 44, f, -22, -22, -22, f, f, -22, -22, f, f, f, f, 54, 46, 48, 60, -22, f, 62, f, -22, -22, 64, 66, -22, -22, f, f, 68, -22, -22, -22, -22, f, f, -22, 50, -22, -22, 56, -22, f, -22, -22, 70, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 55, f, 41, 42 },
    { -30, -30, f, -30, 44, f, -30, -30, -30, f, f, -30, -30, f, f, f, f, -30, 46, 48, -30, -30, f, -30, f, -30, -30, -30, -30, -30, -30, f, f, -30, -30, -30, -30, -30, f, f, -30, 50, -30, -30, 56, -30, f, -30, -30, -30, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 57, f, 41, 42 },
    { -28, -28, f, -28, 44, f, -28, -28, -28, f, f, -28, -28, f, f, f, f, -28, 46, 48, -28, -28, f, -28, f, -28, -28, -28, -28, -28, -28, f, f, -28, -28, -28, -28, -28, f, f, -28, 50, -28, -28, 56, -28, f, -28, -28, -28, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 59, f, 41, 42 },
    { -23, -23, f, -23, -23, f, -23, -23, -23, f, f, -23, -23, f, f, f, f, -23, 46, 48, -23, -23, f, -23, f, -23, -23, -23, -23, -23, -23, f, f, -23, -23, -23, -23, -23, f, f, -23, 50, -23, -23, -23, -23, f, -23, -23, -23, -23 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 61, f, 41, 42 },
    { -31, -31, f, -31, 44, f, -31, -31, -31, f, f, -31, -31, f, f, f, f, -31, 46, 48, -31, -31, f, -31, f, -31, -31, -31, -31, -31, -31, f, f, -31, -31, -31, -31, -31, f, f, -31, 50, -31, -31, 56, -31, f, -31, -31, -31, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 63, f, 41, 42 },
    { -32, -32, f, -32, 44, f, -32, -32, -32, f, f, -32, -32, f, f, f, f, -32, 46, 48, -32, -32, f, -32, f, -32, -32, -32, -32, -32, -32, f, f, -32, -32, -32, -32, -32, f, f, -32, 50, -32, -32, 56, -32, f, -32, -32, -32, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 65, f, 41, 42 },
    { -33, -33, f, -33, 44, f, -33, -33, -33, f, f, -33, -33, f, f, f, f, -33, 46, 48, -33, -33, f, -33, f, -33, -33, -33, -33, -33, -33, f, f, -33, -33, -33, -33, -33, f, f, -33, 50, -33, -33, 56, -33, f, -33, -33, -33, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 67, f, 41, 42 },
    { -34, -34, f, -34, 44, f, -34, -34, -34, f, f, -34, -34, f, f, f, f, -34, 46, 48, -34, -34, f, -34, f, -34, -34, -34, -34, -34, -34, f, f, -34, -34, -34, -34, -34, f, f, -34, 50, -34, -34, 56, -34, f, -34, -34, -34, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 69, f, 41, 42 },
    { -29, -29, f, -29, 44, f, -29, -29, -29, f, f, -29, -29, f, f, f, f, -29, 46, 48, -29, -29, f, -29, f, -29, -29, -29, -29, -29, -29, f, f, -29, -29, -29, -29, -29, f, f, -29, 50, -29, -29, 56, -29, f, -29, -29, -29, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 71, f, 41, 42 },
    { -21, -21, f, -21, 44, f, -21, -21, -21, f, f, -21, -21, f, f, f, f, 54, 46, 48, 60, -21, f, 62, f, -21, -21, 64, 66, -21, -21, f, f, 68, -21, -21, -21, -21, f, f, -21, 50, -21, -21, 56, -21, f, -21, -21, -21, 58 },
    { -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50, -50 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, -60, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, 74, f, 77, f, f, f, f, f, 37, f, f, 38, 79, f, 41, 42 },
    { f, -61, f, f, f, f, -61, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -61, -61, -61, -61, f, f, f, f, f, -61, f, 75 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 76, f, 41, 42 },
    { -65, -65, f, f, 44, f, -65, -65, -65, f, f, 52, -65, f, f, f, f, 54, 46, 48, 60, -65, f, 62, f, -65, -65, 64, 66, -65, -65, f, f, 68, -65, -65, -65, -65, f, f, -65, 50, f, -65, 56, -65, f, f, -65, 70, 58 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 78 },
    { -17, -17, f, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17, f, f, f, -17, -17, -17, -17, -17, f, -17, f, -17, -17, -17, -17, -17, -17, f, f, -17, -17, -17, -17, -17, f, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17, -17 },
    { -64, -64, f, f, 44, f, -64, -64, -64, f, f, 52, -64, f, f, f, f, 54, 46, 48, 60, -64, f, 62, f, -64, -64, 64, 66, -64, -64, f, f, 68, -64, -64, -64, -64, f, f, -64, 50, f, -64, 56, -64, f, f, -64, 70, 58 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 81 },
    { -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49, -49 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 83 },
    { f, f, f, f, f, 35, f, f, f, f, f, f, 73, 10, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 84, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 85 },
    { -19, -19, f, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19, f, f, f, -19, -19, -19, -19, -19, f, -19, f, -19, -19, -19, -19, -19, -19, f, f, -19, -19, -19, -19, -19, f, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19, -19 },
    { -16, -16, f, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16, f, f, f, -16, -16, -16, -16, -16, f, -16, f, -16, -16, -16, -16, -16, -16, f, f, -16, -16, -16, -16, -16, f, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16, -16 },
    { -15, -15, f, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15, f, f, f, -15, -15, -15, -15, -15, f, -15, f, -15, -15, -15, -15, -15, -15, f, f, -15, -15, -15, -15, -15, f, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15, -15 },
    { -38, -38, -38, -38, -38, -38, -38, -38, -38, 39, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38, -38 },
    { -39, -39, -39, -39, -39, -39, -39, -39, -39, 39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39, -39 },
    { f, f, f, f, 44, f, f, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, f, 50, 90, f, 56, f, f, f, f, 70, 58 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 91 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 92, f, 41, 42 },
    { f, f, f, f, 44, f, -94, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, f, 50, f, f, 56, -94, f, -94, f, 70, 58 },
    { -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, 94, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97, -97 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 95, f, 41, 42 },
    { f, f, f, f, 44, f, -93, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, f, 50, f, f, 56, -93, f, -93, f, 70, 58 },
    { f, f, f, f, f, f, 97, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 98, f, -90, f, f, f, f, f, f, 99 },
    { f, f, -95, f, -95, -95, f, f, f, f, -95, f, -95, -95, -95, -95, -95, f, f, f, f, f, -95, f, f, f, -95, f, f, -95, f, -95, f, f, f, f, f, f, -95, f, f, f, f, f, f, f, f, -95 },
    { f, f, -96, f, -96, -96, f, f, f, f, -96, f, -96, -96, -96, -96, -96, f, f, f, f, f, -96, f, f, f, -96, f, f, -96, f, -96, f, f, f, f, f, f, -96, f, f, f, f, f, f, f, f, -96 },
    { f, f, 6, f, 7, 8, f, f, f, f, 11, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 93, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, -89, f, f, f, f, f, 32, f, 96, 100, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 101, f, 41, 42 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -91 },
    { f, f, f, f, 44, f, -92, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, f, 50, f, f, 56, -92, f, -92, f, 70, 58 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 103 },
    { -88, -88, f, -88, -88, -88, -88, -88, -88, -88, -88, -88, -88, -88, f, f, f, -88, -88, -88, -88, -88, f, -88, f, -88, -88, -88, -88, -88, -88, f, f, -88, -88, -88, -88, -88, f, -88, -88, -88, -88, -88, -88, -88, -88, -88, -88, -88, -88 },
    { f, f, f, f, 44, f, f, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, f, 50, f, 105, 56, f, f, f, f, 70, 58 },
    { -71, -71, f, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71, f, f, f, -71, -71, -71, -71, -71, f, -71, f, -71, -71, -71, -71, -71, -71, f, f, -71, -71, -71, -71, -71, f, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71, -71 },
    { -37, -37, -37, -37, -37, -37, -37, -37, -37, 39, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37, -37 },
    { f, f, f, f, 44, f, f, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, 108, 50, f, f, 56, f, f, f, f, 70, 58 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 109, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 110 },
    { -75, -75, f, f, f, f, -75, -75, -75, f, f, f, -75, f, f, f, f, f, f, f, f, -75, f, f, f, -75, -75, f, f, -75, -75, f, f, f, -75, -75, -75, -75, f, f, -75, f, f, f, f, f, f, f, -75 },
    { f, -60, 6, f, 7, 8, -60, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, -60, -60, -60, -60, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, 74, f, 112, f, f, f, f, f, 37, f, f, 38, 79, f, 41, 42 },
    { f, -44, f, f, f, f, 113, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -44, -44, -44, -44 },
    { f, -45, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -45, -45, -45, -45 },
    { -53, -53, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 115, f, f, f, f, f, f, f, f, f, 4 },
    { f, 116 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 117, f, 41, 42 },
    { -76, -76, f, f, 44, f, -76, -76, -76, f, f, 52, -76, f, f, f, f, 54, 46, 48, 60, -76, f, 62, f, -76, -76, 64, 66, -76, -76, f, f, 68, -76, -76, -76, -76, f, f, -76, 50, f, f, 56, f, f, f, -76, 70, 58 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, f, 119, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 124, f, f, f, f, f, f, f, 125 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 120 },
    { f, f, f, f, f, f, f, f, f, f, f, f, 18, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 121 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 122, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 123 },
    { -84, -84, f, f, f, f, -84, -84, -84, f, f, f, -84, f, f, f, f, f, f, f, f, -84, f, f, f, -84, -84, f, f, -84, -84, f, f, f, -84, -84, -84, -84, f, f, -84, f, f, f, f, f, f, f, -84 },
    { -68, -68, f, f, f, f, -68, -68, -68, f, f, f, -68, f, f, f, f, f, f, f, f, -68, f, f, f, -68, -68, f, f, -68, -68, f, -68, f, -68, -68, -68, -68, f, f, -68, f, f, f, f, -68, f, f, -68 },
    { -82, -82, f, f, f, f, -82, -82, -82, f, f, f, -82, f, f, f, f, f, f, f, f, -82, f, f, f, -82, -82, f, f, -82, -82, f, 126, f, -82, -82, -82, -82, f, f, -82, f, f, f, f, 128, f, f, -82 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, 127, f, f, f, f, f, f, f, 37, f, f, 38, 79, f, 41, 42 },
    { -83, -83, f, f, f, f, -83, -83, -83, f, f, f, -83, f, f, f, f, f, f, f, f, -83, f, f, f, -83, -83, f, f, -83, -83, f, f, f, -83, -83, -83, -83, f, f, -83, f, f, f, f, 75, f, f, -83 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 129 },
    { -69, -69, f, f, f, f, -69, -69, -69, f, f, f, -69, f, f, f, f, f, f, f, f, -69, f, f, -69, -69, -69, f, f, -69, -69, f, -69, f, -69, -69, -69, -69, f, f, -69, f, f, f, f, -69, f, f, -69 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 131, f, 41, 42 },
    { f, f, f, 132, 44, f, f, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, f, 50, f, f, 56, f, f, f, f, 70, 58 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, -53, -53, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 133, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -56, -56, -56, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 134 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 135, 136, 140 },
    { -77, -77, f, f, f, f, -77, -77, -77, f, f, f, -77, f, f, f, f, f, f, f, f, -77, f, f, f, -77, -77, f, f, -77, -77, f, f, f, -77, -77, -77, -77, f, f, -77, f, f, f, f, f, f, f, -77 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 137, f, 41, 42 },
    { f, f, f, 138, 44, f, f, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, f, 50, f, f, 56, f, f, f, f, 70, 58 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, -53, -53, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 139, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -57, -57, -57 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 141, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 142 },
    { -78, -78, f, f, f, f, -78, -78, -78, f, f, f, -78, f, f, f, f, f, f, f, f, -78, f, f, f, -78, -78, f, f, -78, -78, f, f, f, -78, -78, -78, -78, f, f, -78, f, f, f, f, f, f, f, -78 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 144, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 145 },
    { f, f, f, f, f, f, f, f, f, f, f, f, -40, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -40, f, f, f, f, f, f, -40 },
    { f, f, f, f, f, f, f, f, f, f, f, f, 18, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 146, f, f, f, f, f, f, 148, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 150 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 147 },
    { f, f, f, f, f, f, f, f, f, f, f, f, -41, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -41, f, f, f, f, f, f, -41 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 149 },
    { f, f, f, f, f, f, f, f, f, f, f, f, -42, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -42, f, f, f, f, f, f, -42 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 151, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 152 },
    { -85, -85, f, f, f, f, -85, -85, -85, f, f, f, -85, f, f, f, f, f, f, f, f, -85, f, f, f, -85, -85, f, f, -85, -85, f, f, f, -85, -85, -85, -85, f, f, -85, f, f, f, f, f, f, f, -85 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 154, f, f, f, f, f, f, f, 167 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -68, f, f, f, f, f, f, f, 155, f, f, f, f, f, f, f, f, f, f, f, f, -68 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 156, f, 41, 42 },
    { f, f, f, f, 44, f, f, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, f, 50, f, f, 56, 157, f, f, f, 70, 58 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 158, f, 41, 42 },
    { f, f, f, f, 44, f, f, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, 159, 50, f, f, 56, 162, f, f, f, 70, 58 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 160, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 161 },
    { -79, -79, f, f, f, f, -79, -79, -79, f, f, f, -79, f, f, f, f, f, f, f, f, -79, f, f, f, -79, -79, f, f, -79, -79, f, f, f, -79, -79, -79, -79, f, f, -79, f, f, f, f, f, f, f, -79 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, f, f, f, f, f, f, f, f, 37, f, f, 38, 163, f, 41, 42 },
    { f, f, f, f, 44, f, f, f, f, f, f, 52, f, f, f, f, f, 54, 46, 48, 60, f, f, 62, f, f, f, 64, 66, f, f, f, f, 68, f, f, f, f, f, f, 164, 50, f, f, 56, f, f, f, f, 70, 58 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 165, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 166 },
    { -80, -80, f, f, f, f, -80, -80, -80, f, f, f, -80, f, f, f, f, f, f, f, f, -80, f, f, f, -80, -80, f, f, -80, -80, f, f, f, -80, -80, -80, -80, f, f, -80, f, f, f, f, f, f, f, -80 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 168, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 128 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, 169, f, f, f, f, f, f, f, 37, f, f, 38, 79, f, 41, 42 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 170, f, f, f, f, 75 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 171, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 172 },
    { -81, -81, f, f, f, f, -81, -81, -81, f, f, f, -81, f, f, f, f, f, f, f, f, -81, f, f, f, -81, -81, f, f, -81, -81, f, f, f, -81, -81, -81, -81, f, f, -81, f, f, f, f, f, f, f, -81 },
    { -53, f, f, f, f, f, f, -53, -53, f, f, f, -53, f, f, f, f, f, f, f, f, -53, f, f, f, -53, -53, f, f, -53, -53, f, f, f, f, -53, f, f, f, f, -53, f, f, f, f, f, f, f, -53, f, f, f, f, f, f, f, f, f, 174, f, f, f, f, f, f, f, f, f, 4 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 175 },
    { -73, -73, f, f, f, f, -73, -73, -73, f, f, f, -73, f, f, f, f, f, f, f, f, -73, f, f, f, -73, -73, f, f, -73, -73, f, f, f, -73, -73, -73, -73, f, f, -73, f, f, f, f, f, f, f, -73 },
    { f, -46, f, f, f, f, 177, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -46, -46, -46, -46 },
    { f, -47, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -47, -47, -47, -47 },
    { -54, -54, f, f, f, f, 179, -54, -54, f, f, f, -54, f, f, f, f, f, f, f, f, -54, f, f, f, -54, -54, f, f, -54, -54, f, f, f, -54, -54, -54, -54, f, f, -54, f, f, f, f, f, f, f, -54 },
    { -55, -55, f, f, f, f, f, -55, -55, f, f, f, -55, f, f, f, f, f, f, f, f, -55, f, f, f, -55, -55, f, f, -55, -55, f, f, f, -55, -55, -55, -55, f, f, -55, f, f, f, f, f, f, f, -55 },
    { f, f, f, f, f, 35, f, f, f, f, 36, f, 73, 10, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 80, f, f, f, f, f, f, 82, f, f, f, f, f, f, f, f, f, f, 84, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 86 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 182, f, f, f, f, f, f, f, f, f, f, f, f, 184 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, 183, f, f, f, f, f, f, f, 37, f, f, 38, 79, f, 41, 42 },
    { -74, -74, f, f, f, f, -74, -74, -74, f, f, f, -74, f, f, f, f, f, f, f, f, -74, f, f, f, -74, -74, f, f, -74, -74, f, f, f, -74, -74, -74, -74, f, f, -74, f, f, f, f, 75, f, f, -74 },
    { f, f, f, f, f, f, f, f, f, f, f, f, 9, f, f, f, f, f, f, f, f, f, f, f, f, f, 16, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, f, f, f, 180, f, f, f, f, f, f, f, f, f, f, 185, f, f, f, f, f, 41 },
    { f, f, f, f, f, -70, f, f, f, f, -70, f, -70, -70, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -67, f, f, f, f, f, f, -70, f, f, f, f, f, -67, -70 },
    { f, f, f, f, f, -70, f, f, f, f, -70, f, -70, -70, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -66, f, f, f, f, f, f, -70, f, f, f, f, f, -66, -70 },
    { -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87, -87 },
    { -86, -86, f, f, f, -72, -86, -86, -86, f, -72, f, -72, -72, f, f, f, f, f, f, f, -86, f, f, f, -86, -86, f, f, -86, -86, f, f, f, -86, -86, -86, -86, f, -72, -86, f, f, f, f, f, -72, f, -86 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -1 },
    { f, f, 6, f, 7, 8, f, f, f, f, f, f, 9, 10, 12, 13, 14, f, f, f, f, f, 15, f, f, f, 16, f, f, 17, f, 30, f, f, f, f, f, f, 31, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 32, f, f, f, 33, f, f, 34, f, f, 191, f, f, f, f, f, 193, f, 37, f, f, 38, 79, f, 41, 42 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, 192, f, f, f, f, f, f, f, f, f, f, 75 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -52 },
    { f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, f, -2 },
}

--- Run the parser across a sequence of tokens.
--
-- @tparam table context The current parser context.
-- @tparam function get_next A stateful function which returns the next token.
-- @treturn boolean Whether the parse succeeded or not.
local function parse(context, get_next, start)
    local stack, stack_n = { start or 1, 1, 1 }, 1
    local reduce_stack = {}

    while true do
        local token, token_start, token_end = get_next()
        local state = stack[stack_n]
        local action = transitions[state][token]

        if not action then -- Error
            return handle_error(context, stack, stack_n, token, token_start, token_end)
        elseif action >= 0 then -- Shift
            stack_n = stack_n + 3
            stack[stack_n], stack[stack_n + 1], stack[stack_n + 2] = action, token_start, token_end
        elseif action >= -2 then -- Accept
            return true
        else -- Reduce
            -- Reduction is quite complex to get right, as the error code expects the parser
            -- to be shifting rather than reducing. Menhir achieves this by making the parser
            -- stack be immutable, but that's hard to do efficiently in Lua: instead we track
            -- what symbols we've pushed/popped, and only perform this change when we're ready
            -- to shift again.

            local popped, pushed = 0, 0
            while true do
                -- Look at the current item to reduce
                local reduce = productions[-action]
                local terminal, to_pop = reduce[1], reduce[2]

                -- Find the state at the start of this production. If to_pop == 0
                -- then use the current state.
                local lookback = state
                if to_pop > 0 then
                    pushed = pushed - to_pop
                    if pushed <= 0 then
                        -- If to_pop >= pushed, then clear the reduction stack
                        -- and consult the normal stack.
                        popped = popped - pushed
                        pushed = 0
                        lookback = stack[stack_n - popped * 3]
                    else
                        -- Otherwise consult the stack of temporary reductions.
                        lookback = reduce_stack[pushed]
                    end
                end

                state = transitions[lookback][terminal]
                if not state or state <= 0 then error("reduce must shift!") end

                -- And fetch the next action
                action = transitions[state][token]

                if not action then -- Error
                    return handle_error(context, stack, stack_n, token, token_start, token_end)
                elseif action >= 0 then -- Shift
                    break
                elseif action >= -2 then -- Accept
                    return true
                else
                    pushed = pushed + 1
                    reduce_stack[pushed] = state
                end
            end

            if popped == 1 and pushed == 0 then
                -- Handle the easy case: Popped one item and replaced it with another
                stack[stack_n] = state
            else
                -- Otherwise pop and push.
                -- FIXME: The positions of everything here are entirely wrong.
                local end_pos = stack[stack_n + 2]
                stack_n = stack_n - popped * 3
                local start_pos = stack[stack_n + 1]

                for i = 1, pushed do
                    stack_n = stack_n + 3
                    stack[stack_n], stack[stack_n + 1], stack[stack_n + 2] = reduce_stack[i], end_pos, end_pos
                end

                stack_n = stack_n + 3
                stack[stack_n], stack[stack_n + 1], stack[stack_n + 2] = state, start_pos, end_pos
            end

            -- Shift the token onto the stack
            stack_n = stack_n + 3
            stack[stack_n], stack[stack_n + 1], stack[stack_n + 2] = action, token_start, token_end
        end
    end
end

return {
    tokens = tokens,
    parse = parse,
    repl_exprs = 190, --[[- The repl_exprs starting state. ]]
    program = 1, --[[- The program starting state. ]]
}
